package org.msh.etbm.services.mobile.init;

import org.jboss.seam.Component;
import org.msh.etbm.rest.mobile.SyncFromServerRequest;
import org.msh.etbm.services.mobile.InitAndSyncDataGenerator;
import org.msh.etbm.services.mobile.model.ExamCultureData;
import org.msh.etbm.services.mobile.model.ExamHIVData;
import org.msh.etbm.services.mobile.model.ExamMicroscopyData;
import org.msh.etbm.services.mobile.model.ExamXpertData;
import org.msh.tb.entities.ExamMicroscopy;
import org.msh.tb.entities.Tbunit;
import org.msh.tb.entities.enums.*;

import javax.persistence.EntityManager;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Return the microscopy exams to the client side
 * Created by rmemoria on 7/5/17.
 */
public class Step4InitGenerator implements InitAndSyncDataGenerator<CaseData1Response> {

    @Override
    public CaseData1Response generate(Tbunit unit) {
        return generate(unit, null);
    }

    @Override
    public CaseData1Response generate(Tbunit unit, SyncFromServerRequest req) {
        CaseData1Response resp = new CaseData1Response();

        resp.setExamsCulture(generateCultureResults(unit, req));

        resp.setExamsMicroscopy(generateMicroscopyResults(unit, req));

        resp.setExamsXpert(generateXpertResults(unit, req));

        resp.setExamsHIV(generateExamsHIV(unit, req));

        return resp;
    }

    private EntityManager getEntityManager() {
        return (EntityManager) Component.getInstance("entityManager");
    }


    private List<ExamCultureData> generateCultureResults(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select e.id, e.tbcase.id, e.dateCollected, e.dateRelease," +
                "e.result, e.comments, e.sampleNumber, e.sampleType, e.method.id, e.laboratory.id, e.numberOfColonies " +
                "from ExamCulture e " +
                "where e.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and e.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and e.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.size() == 0) {
            return null;
        }

        List<ExamCultureData> res = new ArrayList<ExamCultureData>(lst.size());

        for (Object[] val: lst) {
            ExamCultureData data = new ExamCultureData();
            data.setServerId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            data.setDate((Date)val[2]);
            data.setDateRelease((Date)val[3]);
            data.setResult((CultureResult)val[4]);
            data.setComments((String)val[5]);
            data.setSampleNumber((String)val[6]);
            data.setSampleType((SampleType)val[7]);
            data.setMethod((Integer)val[8]);
            data.setLaboratoryId((Integer)val[9]);
            data.setNumberOfColonies((Integer)val[10]);

            res.add(data);
        }

        return res;
    }

    /**
     * Generate microscopy results to send to client
     * @param unit
     * @return
     */
    private List<ExamMicroscopyData> generateMicroscopyResults(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "from ExamMicroscopy e join fetch e.tbcase " +
                "where e.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and e.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and e.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<ExamMicroscopy> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<ExamMicroscopyData> res = new ArrayList<ExamMicroscopyData>();

        for (ExamMicroscopy exam: lst) {
            res.add(convertData(exam));
        }
        return res;
    }

    /**
     * Generate Xpert result data
     * @param unit
     * @return
     */
    private List<ExamXpertData> generateXpertResults(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select e.id, e.result, e.rifResult, e.tbcase.id, " +
                "e.dateCollected, e.dateRelease, e.sampleNumber, e.comments, e.laboratory.id " +
                "from ExamXpert e " +
                "where e.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and e.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and e.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.isEmpty()) {
            return null;
        }

        List<ExamXpertData> res = new ArrayList<ExamXpertData>(lst.size());

        for (Object[] vals: lst) {
            ExamXpertData data = new ExamXpertData();
            data.setServerId((Integer)vals[0]);
            XpertResult result = (XpertResult)vals[1];
            XpertRifResult rifResult = (XpertRifResult)vals[2];
            ExamXpertData.Result dataResult = convertXpertResultToResult(result, rifResult);
            data.setResult(dataResult);
            data.setCaseId((Integer)vals[3]);
            data.setDate((Date) vals[4]);
            data.setDateRelease((Date) vals[5]);
            data.setSampleNumber((String)vals[6]);
            data.setComments((String)vals[7]);
            data.setLaboratoryId((Integer)vals[8]);

            res.add(data);
        }

        return res;
    }

    private ExamXpertData.Result convertXpertResultToResult(XpertResult result, XpertRifResult rifResult) {
        switch (result) {
            case INVALID:
            case NO_RESULT:
            case ERROR:
                return ExamXpertData.Result.INVALID_NORESULT_ERROR;
            case TB_NOT_DETECTED:
                return ExamXpertData.Result.TB_NOT_DETECTED;
            case TB_DETECTED: {
                switch (rifResult) {
                    case RIF_DETECTED:
                        return ExamXpertData.Result.TB_DETECTED_RR_DETECTED;
                    case RIF_NOT_DETECTED:
                        return ExamXpertData.Result.TB_DETECTED_RR_NOT_DETECTED;
                    case RIF_INDETERMINATE:
                        return ExamXpertData.Result.TB_DETECTED_RR_INDETERMINATE;
                }
            }
            case INVALID_NORESULT_ERROR:
                return ExamXpertData.Result.INVALID_NORESULT_ERROR;
        }

        throw new RuntimeException("Invalid result value: " + result + ", " + rifResult);
    }

    private ExamMicroscopyData convertData(ExamMicroscopy exam) {
        ExamMicroscopyData data = new ExamMicroscopyData();
        data.setServerId(exam.getId());
        data.setDate(exam.getDateCollected());
        data.setCaseId(exam.getTbcase().getId());
        data.setSampleType(exam.getSampleType());
        data.setSampleNumber(exam.getSampleNumber());
        data.setDateRelease(exam.getDateRelease());
        data.setResult(exam.getResult());
        data.setNumberOfAFB(exam.getNumberOfAFB());
        data.setComments(exam.getComments());
        data.setMethod(exam.getMethod() != null ? exam.getMethod().getId() : null);
        data.setLaboratoryId(exam.getLaboratory() != null ? exam.getLaboratory().getId() : null);
        return data;
    }

    /**
     * Generate exams HIV to be sent to the client
     * @param unit
     * @return
     */
    private List<ExamHIVData> generateExamsHIV(Tbunit unit, SyncFromServerRequest req) {
        String qryString = "select e.id, e.tbcase.id, e.date, e.result, e.cd4Count," +
                "e.cd4StDate, e.startedARTdate, e.startedCPTdate " +
                "from ExamHIV e " +
                "where e.tbcase.ownerUnit.id = :id ";

        if (req != null && req.hasBothIndexes()) {
            qryString += " and e.lastTransaction.id > " + req.getServerIndexFrom();
            qryString += " and e.lastTransaction.id <= " + req.getServerIndexTo();
        }

        List<Object[]> lst = getEntityManager()
                .createQuery(qryString)
                .setParameter("id", unit.getId())
                .getResultList();

        if (lst.size() == 0) {
            return null;
        }

        List<ExamHIVData> res = new ArrayList<ExamHIVData>(lst.size());

        for (Object[] val: lst) {
            ExamHIVData data = new ExamHIVData();
            data.setServerId((Integer)val[0]);
            data.setCaseId((Integer)val[1]);
            data.setDate((Date)val[2]);
            data.setResult((HIVResult)val[3]);
            data.setCd4Count((Integer)val[4]);
            data.setCd4StDate((Date)val[5]);
            data.setStartedARTdate((Date)val[6]);
            data.setStartedCPTdate((Date)val[7]);

            res.add(data);
        }

        return res;
    }
}
